#!/bin/bash
SCRIPT_PATH=$(dirname $(readlink -f ${BASH_SOURCE[0]}))
DISK_BOARD_FW_DIR="$SCRIPT_PATH/../../../firmware/bridge/"
UPDATE_TOOL="$SCRIPT_PATH/../../tools/hdparm/hdparm"
COMMON_PATH="$SCRIPT_PATH/../cmn/common.sh"
PATHCFG_PATH="$SCRIPT_PATH/../../pathcfg.sh"
SMARTCTL_TOOL="/usr/bin/smartctl6.7-linux"
LSI_TOOL="/usr/bin/storcli-linux"
source $COMMON_PATH
source $PATHCFG_PATH

function ListHardDiskDrive()
{
    local ctrlModel=$1
    local disk_info=$2
    local newVersion=$(echo ${disk_info}|awk -F, '{print $1}')
    local historicalVer=$(echo ${disk_info} | awk -F, '{print $5}')
    # 如果历史版本号不为N/A, 在list的时候将历史版本号展示出来，从而区分OEM和代理硬盘
    if [[ "${historicalVer}" != "N/A" ]];then
        local diskInfo="${ctrlModel}, ${historicalVer}, bridge"
    else
        local diskInfo="${ctrlModel}, bridge"
    fi
    if [ ! -f bridge_disk.list ];then
        echo "scaning bridge straight harddisk in server ......"
    fi
    declare -A dev_model_dict
    getAllBridgeDisk ${dev_model_dict}

    rm -f curBridgeDisk.tmp
    for dev in ${!dev_model_dict[*]}
    do
        if [[ "${dev_model_dict["${dev}"]}" =~ "${ctrlModel}" ]];then
            local fwVersion=`${UPDATE_TOOL} -I ${dev} | grep "Firmware Revision:" | awk -F ':' '{print $2}' | sed -e 's/[ \t]*//g'`
            if [ "${historicalVer}" == "N/A" ] || [[ "x${fwVersion}" =~ "x${historicalVer}" ]];then
                echo "${dev},${ctrlModel}" >> curBridgeDisk.tmp
            fi
        fi
    done

    if [ ! -f curBridgeDisk.tmp ];then
        echo "no harddisk is bridge straight in this server" >> /tmp/tmp.log
        return 0
    fi

    while read line
    do
        if [ "${line}" != "" ];then
            dev=$(echo ${line} | awk -F ',' '{print $1}')
            curModel=$(echo ${line} | awk -F ',' '{print $2}')
            if [ "${dev}" != "" ] && [ "${curModel}" != "" ] && [[ "${curModel}" =~ "${ctrlModel}" ]];then
                curVersion=`${UPDATE_TOOL} -I ${dev} | grep "Firmware Revision:" | awk -F ':' '{print $2}' | sed -e 's/[ \t]*//g'`
                sn=`${UPDATE_TOOL} -I ${dev} | grep "Serial Number:" | awk -F ':' '{print $2}' | sed -e 's/[ \t]*//g'`
                if [ "x${curVersion}" = "x${newVersion}" ];then
                    printf "%-10s %-10s %-20s %-10s %-16s %-8s %-3s \n" "harddisk" "--"  ${ctrlModel} "default" ${sn} ${curVersion} "--" >> bridge_disk_tmp.txt
                elif [ "${historicalVer}" = "N/A" ] || [[ "x${curVersion}" =~ "x${historicalVer}" ]];then
                    echo "need update" >> ${NEED_UPDATE_FLAG}
                    printf "\e[31m%-10s %-10s %-20s %-10s %-16s %-8s %-3s \e[0m\n" "harddisk" "--" ${ctrlModel} "default" ${sn} ${curVersion} "--" >> bridge_disk_tmp.txt
                fi
            fi
        fi
    done < curBridgeDisk.tmp
    rm -f curBridgeDisk.tmp

    if [ ! -f "bridge_disk_tmp.txt" ];then
        echo "no harddisk " >> bridge_disk_tmp.txt
    fi
    local TotalNum=$(cat bridge_disk_tmp.txt| grep "$ctrlModel" | wc -l)
    local ModelNum=0
    # ModelNum是为了部分硬盘升级时，需要升级两次，桥接固件版本和目标固件版本，确保同型号硬盘只打印一次信息
    if [ -f bridge_disk.list ];then
        ModelNum=$(cat bridge_disk.list| grep "$diskInfo"|wc -l)
    fi

    if [ ${ModelNum} -eq 0 ];then
        echo -e "====================Harddisk model(${diskInfo}) info===================== " >> bridge_disk.list
        printf "%-10s: %-8s\n"   TotalNum  ${TotalNum}    >> bridge_disk.list
        printf "%-10s %-10s %-20s %-10s %-16s %-8s %-3s\n"   Device  Verdor  Model FwType SerialNum  FwVer Slot >> bridge_disk.list
        cat bridge_disk_tmp.txt | grep "$ctrlModel" >> bridge_disk.list
    fi
    rm -f bridge_disk_tmp.txt
}

function updateDiskFwCheck()
{
    # "result.tmp" "${need_update}" "${ctrlModel}" "${newVersion}" "${updatetime}"
    local resultFile=$1
    local need_update=$2
    local ctrlModel=$3
    local newVersion=$4
    local updatetime=$5
    local diskInfo=$6

    fwSucced=$(cat $resultFile | grep "update OK" | wc -l)
    fwfailed=$(cat $resultFile | grep "update failed" | wc -l)
    if [ $fwSucced -eq $need_update ];then
        greenFont "Total bridge straight harddisk ($diskInfo) need update num:$need_update, fw succeed num: $fwSucced, fw fwfailed num: ${fwfailed}" | tee -a bridge_update_harddisk.result
        updateResult $diskInfo "all" "default" "success" "--" $newVersion $updatetime "bridge straight harddisk"
    else
        redFont "Total bridge straight harddisk ($diskInfo) need update num:$need_update, fw succeed num: $fwSucced, fw fwfailed num: ${fwfailed}, please Check!" | tee -a bridge_update_harddisk.result
        updateResult $diskInfo "all" "default" "failed" "--" $newVersion $updatetime "bridge straight harddisk"
    fi
}

function updateDiskBridgeFwMain()
{
    ctrlModel=$1
    disk_info=$2
    all_dev_model=$3
    local newVersion=$(echo ${disk_info} | awk -F, '{print $1}')
    local fwFile=$(echo ${disk_info} | awk -F, '{print $2}')
    local full_fwfile=${DISK_BOARD_FW_DIR}${fwFile}
    local retry_times=$(echo ${disk_info} | awk -F, '{print $3}')
    local commit_cmd=$(echo ${disk_info} | awk -F, '{print $4}')
    local historical_ver=$(echo ${disk_info} | awk -F, '{print $5}')
    let retry_times=retry_times+1

    if [ ${historical_ver} != "N/A" ];then
        local diskInfo="${ctrlModel}|${historical_ver}"
    else
        local diskInfo="${ctrlModel}"
    fi

    if [ ! -f "${full_fwfile}" ] || [ ! -f ${UPDATE_TOOL} ];then
        redFont "fw update tool(hdparm) or ${fwFile} cat not found, exit upgrade"
        updateResult $diskInfo "--" "default" "failed" "--" "--" "0" "no tool or fwfile"
        return 0
    fi
    chmod +x ${UPDATE_TOOL}

    local total_disk=0
    local need_update=0
    
    starttime=`date +'%Y-%m-%d %H:%M:%S'`
    start_seconds=$(date --date="$starttime" +%s)

    declare -A dev_model_dict
    getAllBridgeDisk ${dev_model_dict}

    rm -f curBridgeDisk.tmp
    for dev in ${!dev_model_dict[*]}
    do
        if [[ "${dev_model_dict["${dev}"]}" =~ "${ctrlModel}" ]];then
            local curVersion=`${UPDATE_TOOL} -I ${dev} | grep "Firmware Revision:" | awk -F ':' '{print $2}' | sed -e 's/[ \t]*//g'`
            # 因为西数OEM和代理硬盘的 型号相同，历史版本号(固件版本前五位)不同
            # 如HUS726T4TALE6L4，OEM为VKGAW9G0，代理为VKGNW9G0，根据前五位进行判断是OEM还是代理渠道的
            # 所以当硬盘历史版本号为N/A，或者当前固件版本和历史固件版本能匹配上时，硬盘总数才+1
            if [ "${historical_ver}" = "N/A" ] || [[ "x${curVersion}" =~ "x${historical_ver}" ]];then
                let total_disk=total_disk+1
                echo "${dev},${ctrlModel}" >> curBridgeDisk.tmp
            fi
        fi
    done

    if [ ${total_disk} -eq 0 ];then
        greenFont "No harddisk(${diskInfo})  is bridge straight in this server, do not need update" | tee -a bridge_update_harddisk.result
        updateResult $diskInfo "--" "default" "success" "--" "--" "0" "no disk is bridge straight"
        return 0
    fi

    while read line
    do
        local dev=$(echo $line | awk -F ',' '{print $1}')
        local ctrlModel=$(echo $line | awk -F ',' '{print $2}')
        local curVersion=`${UPDATE_TOOL} -I ${dev} | grep "Firmware Revision:" | awk -F ':' '{print $2}' | sed -e 's/[ \t]*//g'`
        if [[ "x${newVersion}" != "x${curVersion}" ]];then
            if [[ "x${curVersion}" =~ "x${historical_ver}" ]] || [ "${historical_ver}" = "N/A" ];then
                let need_update=need_update+1
            fi
        fi
        if [ ${need_update} -ne 0 ];then
            local retry_counts=0
            local upgrade_success="False"
            # Tool --fwdownload FW --yes-i-know-what-i-am-doing --please-destroy-my-drive DEV
            # hdparm --fwdownload /root/Desktop/xxx.bin --yes-i-know-what-i-am-doing --please-destroy-my-drive /dev/sdc
            update_cmd=$(echo ${commit_cmd} | sed "s%Tool%${UPDATE_TOOL}%g;s%FW%${full_fwfile}%g;s%DEV%${dev}%g")
            echo "hdparm update firmware cmd: $update_cmd" >> /tmp/tmp.log
            while [ ${retry_counts} -lt ${retry_times} ]
            do
                $update_cmd >> /tmp/tmp.log
                if [ $? -eq 0 ];then
                    echo "Harddisk($diskInfo) update OK, retry_counts:${retry_counts}" >> bridge_harddisk_result.tmp
                    let retry_counts=retry_counts+1
                    upgrade_success="True"
                    break
                else
                    echo "Harddisk($diskInfo) update failed, retry_counts:${retry_counts}" >> /tmp/tmp.log
                    let retry_counts=retry_counts+1
                    sleep 2
                    continue
                fi
            done
            if [ ${upgrade_success} == "False" ];then
                echo "Harddisk($diskInfo) update failed, retry_counts:${retry_counts}" >> bridge_harddisk_result.tmp
            fi
        fi
    done < curBridgeDisk.tmp

    endtime=`date +'%Y-%m-%d %H:%M:%S'`
    end_seconds=$(date --date="$endtime" +%s)
    updatetime=$((end_seconds-start_seconds))
    if [ ${need_update} -eq 0 ];then
        greenFont "Harddisk(${diskInfo}) already newVersion(${newVersion}) or not meet the upgrade criteria, do not need update" | tee -a bridge_update_harddisk.result
        updateResult $diskInfo "All" "default" "success" $newVersion $newVersion $updatetime "already newVer bridge straight harddisk or not meet the upgrade criteria"
        return 0
    fi

    if [ -f "bridge_harddisk_result.tmp" ];then
        updateDiskFwCheck "bridge_harddisk_result.tmp" "${need_update}" "${ctrlModel}" "${newVersion}" "${updatetime}" "${diskInfo}"
        rm -f "bridge_harddisk_result.tmp"
    fi
}

# 获取所有的PMC卡下的硬盘
function getAllPMCDisk()
{
    pmc_sn_model_dict=$1
    # 判断字典的key和value是否都为空, 不为空则表示已经获取过了
    if [ "${pmc_sn_model_dict[*]}" != "" ] && [ "${!pmc_sn_model_dict[*]}" != "" ];then
        return
    fi
    lsscsi -g | grep "Smart Adapter" | awk '{print $NF}' > raid_letter.tmp
    if [ ! -f raid_letter.tmp ];then
        greenFont "No PMC controller in server"
        echo "No PMC controller in server" > ${UPDATE_LOG}
        return 0
    fi
    while read line
    do
        for((i=0;i<65;i++))
        do
            # SATA盘过滤字段为Device Model和Serial Number， SAS盘过滤字段为Product和Serial number
            # 加冒号是防止过滤Device Model的时候，过滤出Vendor Product Id的情况
            model=`${SMARTCTL_TOOL} -a -d cciss,${i} ${line} | grep -E "Device Model:|Product:" | awk -F ':' '{print $2}' |sed -e 's/^[ \t]*//g'`
            sn=`${SMARTCTL_TOOL} -a -d cciss,${i} ${line} | grep -E "Serial Number:|Serial number:" | awk -F ':' '{print $2}' |sed -e 's/^[ \t]*//g'`
            if [ "${line}" != "" ] && [ "${model}" != "" ] && [ "${sn}" != "" ];then
                pmc_sn_model_dict["${sn}"]="${model}"
            fi
        done
    done < raid_letter.tmp
    
}

# 获取所有的LSI卡下的硬盘
function getAllLSIDisk()
{
    lsi_sn_model_dict=$1
    if [ "${lsi_sn_model_dict[*]}" != "" ] && [ "${!lsi_sn_model_dict[*]}" != "" ];then
        return
    fi

    ${LSI_TOOL} show | grep -E "AVAGOMegaRAIDSAS|MegaRAID|SAS|AVAGOJBOD" > lsi_info.tmp
    if [ ! -f lsi_info.tmp ];then
        greenFont "No LSI controller in server"
        echo "No LSI controller in server" > ${UPDATE_LOG}
        return 0
    fi

    while read line
    do
        # SAS兼容9400、9500, MegaRAID兼容9540、9560
        local flag=$(echo $line |grep -Ec "AVAGOMegaRAIDSAS|MegaRAID|SAS|AVAGOJBOD")
        if [ $flag -ne 0 ];then
            local cardid=$(echo $line | sed -e 's/^[ \t]*//g'| awk '{print $1}')
            ${LSI_TOOL} /c$cardid /eall /sall show | grep -E "SATA|SAS" > lsi_disk_info.tmp
            if [ -f lsi_disk_info.tmp ];then
                while read disk
                do
                    local eid=$(echo $disk | awk '{print $1}'| awk -F: '{print $1}')
                    local slotid=$(echo $disk | awk '{print $1}'| awk -F: '{print $2}')
                    local sn=$(${LSI_TOOL} /c$cardid /e$eid /s$slotid show all |grep "SN" | awk '{print $3}')
                    local model=$(${LSI_TOOL} /c$cardid /e$eid /s$slotid show all |grep "Model Number" | awk -F '=' '{print $2}' | sed -e 's/^[ ]//')
                    if [ "${sn}" != "" ] && [ "${model}" != "" ];then
                        lsi_sn_model_dict["${sn}"]="${model}"
                    fi
                done < lsi_disk_info.tmp
            fi
        fi
    done < lsi_info.tmp
}

# raid卡为JBOD模式、raid卡下硬盘状态为JBOD时，hdparm也能读取到硬盘信息，所以通过hdparm获取所有可获取的硬盘信息后，排除挂载raid卡下硬盘信息
function getAllBridgeDisk()
{
    dev_model_dict=$1
    declare -A pmc_sn_model_dict
    getAllPMCDisk ${pmc_sn_model_dict}

    declare -A lsi_sn_model_dict
    getAllLSIDisk ${lsi_sn_model_dict}

    declare -A all_dev_model
    declare -A sn_dev_dict
    if [ "${all_dev_model[*]}" != "" ] && [ "${!all_dev_model[*]}" != "" ] && [ "${sn_dev_dict[*]}" != "" ] && [ "${!sn_dev_dict[*]}" != "" ];then
        return
    fi
    ${SMARTCTL_TOOL} --scan | grep -v "megaraid" > disk_letters.tmp
    if [ -f disk_letters.tmp ];then
        while read line
        do
            if [ "${line}" != "" ];then
                # 过滤megaraid是排除普通lsi卡下挂的硬盘信息
                dev=$(echo ${line} | awk '{print $1}')
                ${UPDATE_TOOL} -I ${dev} >& diskinfo.tmp
                curModel=$(cat diskinfo.tmp | grep "Model Number:" | awk -F ':' '{print $2}' | sed -e 's/^[ \t]*//g' | sed -e 's/[ ]*$//g')
                sn=$(cat diskinfo.tmp | grep "Serial Number:" | awk -F ':' '{print $2}' | sed -e 's/^[ \t]*//g' | sed -e 's/[ ]*$//g')
                if [ "${curModel}" != "" ] && [ "${dev}" != "" ] && [ "${sn}" != "" ];then
                    sn_dev_dict["${sn}"]="${dev}"
                    all_dev_model["${sn}"]="${curModel}"
                fi
            fi
        done < disk_letters.tmp
        rm -f disk_letters.tmp
    fi

    echo "bridge straight harddisk list:" >> ${UPDATE_LOG}
    for key in ${!all_dev_model[*]}
    do
        if [ -z "${pmc_sn_model_dict["${key}"]}" ] && [ -z "${lsi_sn_model_dict["${key}"]}" ] && [ ! -z "${sn_dev_dict["${key}"]}" ];then
            if [ "${sn_dev_dict["${key}"]}" != "" ] && [ "${all_dev_model["${key}"]}" != "" ];then
                dev_model_dict["${sn_dev_dict["${key}"]}"]="${all_dev_model["${key}"]}"
                echo "dev is ${sn_dev_dict["${key}"]}, model is ${all_dev_model["${key}"]}" >> ${UPDATE_LOG}
            fi
        fi
    done
}

function main()
{
    local harddisk_board_cfg=$1
    local upgrade_flag=""
    if [ $# -eq 1 ];then
        upgrade_flag="true"
    elif [ $# -gt 1 ] && [ $2 == "list" ];then
        upgrade_flag="false"
    else
        echo "para error return now"
        exit 1
    fi
    
    declare -a disk_seq
    dosToUnix $harddisk_board_cfg
    declare -A cfg_dict

    while read LINE
    do
       line=$LINE
       if [[ $LINE == \#* ]];then
           continue
       fi
       OLD_IFS="$IFS"
       IFS=","
       configInfo=($LINE)
       IFS="$OLD_IFS"
       model=$(echo ${configInfo[0]})
       new_version=$(echo ${configInfo[1]})
       fwfile=$(echo ${configInfo[2]})
       retry_times=$(echo ${configInfo[3]})
       upgrade_cmd=$(echo ${configInfo[4]})
       historical_version=$(echo ${configInfo[5]})
       if [[ "${historical_version}" != "N/A" ]];then
           model="${model}|${historical_version}"
       fi
       disk_seq+=(${model})
       cfg_dict["${model}"]="${new_version},${fwfile},${retry_times},${upgrade_cmd},${historical_version}"
    done < ${harddisk_board_cfg}

    currsegment=0
    totalnum=${#disk_seq[*]}
    totalsegment=`expr $totalnum \* 2 `

    for disk_model in ${disk_seq[*]}
    do
        local ctrlModel=$(echo ${disk_model} | awk -F "|" '{print $1}')
        if [ ${upgrade_flag} == "true" ];then
            let currsegment=$currsegment+1
            progressinfo=`echo ${ctrlModel} default updating`
            updateDiskBridgeFwMain "${ctrlModel}" "${cfg_dict["${disk_model}"]}"
            let currsegment=$currsegment+1
            progressinfo=`echo ${ctrlModel} default update finished`
        else
            ListHardDiskDrive "${ctrlModel}" "${cfg_dict["${disk_model}"]}"
        fi
    done
}

main $@