#!/bin/bash

NVME_TOOL_NAME="nvme"

##===================================#
# 功能：SAMSUNG disk信息查询
# 入参：
#       $1: ctrlModel  待升级固件型号
#       $2: diskInfoFile
#       $3: fwNewVer  待升级固件版本
#       $4: fwType 待升级固件类型
##===================================#
function listSAMSUNGNvmeDiskDetails()
{
    local ctrlModel=$1
    local diskInfoFile=$2
    local fwNewVer=$3
    local fwType=$4

    if [ ${fwType} = "fw" ]
    then
        printf "%-10s %-10s %-20s %-10s %-16s %-8s %-3s\n"   Device  Verdor  Model FwType SerialNum  FwVer Slot >> samsung_nvme.list
    fi
    OLD_IFS="$IFS"
    IFS=$'\n'
    while read myline
    do
        local disk_model=$(echo ${myline} | awk '{print $1}')
        if [ "x${disk_model}" = "x${ctrlModel}" ]
        then
            local FW_Rev=$(echo ${myline} | awk '{print $NF}')
            if [ "${FW_Rev}" != "${fwNewVer}" ]
            then
                echo "need update" >> ${NEED_UPDATE_FLAG}
                printf "\e[31m%-10s %-10s %-20s %-10s %-16s %-8s %-3s \e[0m\n" "harddisk" "SAMSUNG" ${ctrlModel} ${fwType} "--"  "${FW_Rev}" "--"  >> samsung_nvme.list
            else
                printf "%-10s %-10s %-20s %-10s %-16s %-8s %-3s \n" "harddisk" "SAMSUNG" ${ctrlModel} ${fwType} "--" "${FW_Rev}" "--"  >> samsung_nvme.list
            fi
        fi
    done < $diskInfoFile
    IFS="${OLD_IFS}"
}

##===================================#
# 功能：SAMSUNG disk信息查询
# 入参：
#       $1: ctrlModel  待升级固件型号
#       $2: fwNewVer  待升级固件版本
#       $3: fwType 待升级固件类型
##===================================#
function listSAMSUNGDiskDrive()
{
    local ctrlModel=$1
    local fwNewVer=$2
    local fwType=$3

    if [ ! -f "SAMSUNG_nvme_disk.tmp" ]
    then
        echo "scanning the samsung nvme disk in server......"
        ${SAMSUNG_NVME_TOOL} list | grep "/dev/nvme" > SAMSUNG_nvme_disk.tmp
    fi

    rm -f list_nvme_disk_info.tmp

    declare -A map
    while read line
    do
        flag=$(echo ${line} | grep -c "${ctrlModel}")
        if [ ${flag} -ne 0 ]
        then
            local Node=$(echo ${line} | awk '{print $1}')
            local FW_Rev=$(echo ${line} | awk '{print $NF}')
            local node_name_simple=$(echo ${Node} | awk -Fn '{print $2}')
            result=${map["${node_name_simple}"]}
            if [ -z "${result}" ]
            then
                map["${node_name_simple}"]="no exist"
                echo "${ctrlModel}" "${Node}" "${FW_Rev}" >> list_nvme_disk_info.tmp
            fi
        fi
    done < SAMSUNG_nvme_disk.tmp

    if [ ! -f "list_nvme_disk_info.tmp" ]
    then
        echo "No this model disk found" > list_nvme_disk_info.tmp
    fi

    local TotalNum=`cat list_nvme_disk_info.tmp | grep ${ctrlModel} | wc -l`

    if [ ${fwType} = "fw" ]
    then
        echo -e "====================SAMSUNG nvme disk model(${ctrlModel}) info===================== " >>samsung_nvme.list
        printf "%-10s: %-8s\n"   TotalNum  ${TotalNum}    >>samsung_nvme.list
    fi
    if [ ${TotalNum} -ne 0 ]
    then
        listSAMSUNGNvmeDiskDetails "${ctrlModel}" "list_nvme_disk_info.tmp" "${fwNewVer}" "${fwType}"
    fi
}


##===================================#
# 功能：SAMSUNG disk信息查询
# 入参："samsung_nvme.tmp" "$fwType" "$need_update_num" "$ctrlModel"
#       $1: samsung_nvme.tmp  升级结果文件
#       $2: fwType  待升级固件类型
#       $3: need_update_num  需要升级硬盘的个数
#       $4: ctrlModel  待升级固件型号
#       $5: fwNewVer 待升级固件版本
#       $6: updateTime 升级所用的时间
##===================================#
function checkSAMSUMGNvmeUpdate()
{
    local samsungNvmeResFile=$1
    local fwType=$2
    local needUpdateNum=$3
    local ctrlModel=$4
    local fwNewVer=$5
    local updateTime=$6

    local fwSuccedNum=$(cat ${samsungNvmeResFile} | grep "fw update OK" | wc -l)
    local fwFailedNum=$(cat ${samsungNvmeResFile} | grep "fw update failed" | wc -l)

    if [ ${fwType} = "fw" ]
    then
        if [ ${fwSuccedNum} -eq ${needUpdateNum} ]
        then
            greenFont "Total harddisk(${ctrlModel}) need update num:${needUpdateNum}, fw succeed num: ${fwSuccedNum}, failed num: ${fwFailedNum}" | tee -a samsung_nvme.result
            updateResult "${ctrlModel}" "all" "${fwType}" "success" "--" "${fwNewVer}" "${updateTime}"
        else
            redFont "Total harddisk(${ctrlModel}) need update num:${needUpdateNum}, fw succeed num: ${fwSuccedNum}, failed num: ${fwFailedNum}, please Check" | tee -a samsung_nvme.result
            updateResult "${ctrlModel}" "all" "${fwType}" "failed" "--" "${fwNewVer}" "${updateTime}"
        fi
    fi
}


##===================================#
# 功能：SAMSUNG disk的固件升级
# 入参：
#       $1: ctrlModel  待升级固件型号
#       $2: fwFile     待升级固件文件
#       $3: fwNewVer   待升级固件版本
#       $4: fwType     待升级固件类型
##===================================#
function updateSAMSUNGDiskFW()
{
    local ctrlModel=$1
    local fwFile=$2
    local fwNewVer=$3
    local fwType=$4

    rm -f "update_nvme_disk_info.tmp"
    chmod +x ${SAMSUNG_NVME_TOOL}

    starttime=`date +'%Y-%m-%d %H:%M:%S'`
    start_seconds=$(date --date="$starttime" +%s)

    ${SAMSUNG_NVME_TOOL} list | grep "/dev/nvme" > total_nvme_disk_info.tmp
    declare -A map
    while read line
    do
        flag=$(echo ${line} | grep -c "${ctrlModel}")
        if [ ${flag} -ne 0 ]
        then
            local Node=$(echo ${line} | awk '{print $1}')
            local FW_Rev=$(echo ${line} | awk '{print $NF}')
            local node_name_simple=$(echo ${Node} | awk -Fn '{print $2}')
            result=${map["${node_name_simple}"]}
            if [ -z "${result}" ]
            then
                map["${node_name_simple}"]="no exist"
                echo "${ctrlModel}" "${Node}" "${FW_Rev}" >> update_nvme_disk_info.tmp
            fi
        fi
    done < total_nvme_disk_info.tmp

    if [ ! -f "update_nvme_disk_info.tmp" ]
    then
        echo -e "Total harddisk (${ctrlModel}) num: 0, success num: 0" | tee -a samsung_nvme.result
        greenFont "No harddisk (${ctrlModel}) in this server, no need to update" | tee -a samsung_nvme.result
        updateResult "${ctrlModel}" "--" "${fwType}" "success" "--" "--" "0" "no disk"
        return 0
    fi

    total_num=$(cat update_nvme_disk_info.tmp | grep -c "${ctrlModel}")
    local already_new_num=0
    local need_update_num=0
    while read myline
    do
        local Node=$(echo ${myline} | awk '{print $2}')
        local FW_Rev=$(echo ${myline} | awk '{print $3}')
        if [ ${fwType} = "fw" ]
        then
            if [ "x${FW_Rev}" = "x${fwNewVer}" ]
            then
                let already_new_num=already_new_num+1
            else
                ${SAMSUNG_NVME_TOOL} fw-download ${Node} -f ${fwFile}
                if [ $? -ne 0 ]
                then
                    echo "Harddisk(${ctrlModel}) ${fwType} update failed, please Check" >> samsung_nvme.tmp
                else
                    ${SAMSUNG_NVME_TOOL} fw-commit ${Node} -s 2 -a 3
                    if [ $? -ne 0 ]
                    then
                        echo "Harddisk(${ctrlModel}) ${fwType} update failed, please Check" >> samsung_nvme.tmp
                    else
                        echo "Harddisk(${vtrlModel}) ${fwType} update OK" >> samsung_nvme.tmp
                    fi
                fi
            fi
        fi
    done < update_nvme_disk_info.tmp

    let need_update_num=total_num-already_new_num

    endtime=`date +'%Y-%m-%d %H:%M:%S'`
    end_seconds=$(date --date="$endtime" +%s)
    updateTime=$((end_seconds-start_seconds))

    if [ ${need_update_num} -eq 0 ]
    then
        greenFont "All harddisk(${ctrlModel}) $fwType is new version(${fwNewVer}) in this server, no need to update" | tee -a samsung_nvme.result
        updateResult "${ctrlModel}" "all" "${fwType}" "success" "${fwNewVer}" "${fwNewVer}" "${updateTime}"
        return 0
    fi

    if [ -f "samsung_nvme.tmp" ]
    then
        checkSAMSUMGNvmeUpdate "samsung_nvme.tmp" "${fwType}" "${need_update_num}" "${ctrlModel}" "${fwNewVer}" "${updateTime}"
        rm -f "samsung_nvme.tmp"
    fi
}

##===================================#
# 功能：SAMSUNG disk的固件升级入口
# 入参：配置文件 各列
#     $1  deviceType 设备类型
#     $2  vendorType 厂商
#     $3  ctrlModel  固件型号
#     $4  fwType     待升级固件类型
#     $5  fwVer      固件版本号
#     $6  fwToolsDir 固件升级的工具路径
#     $7  fwFiles    固件的版本文件
##===================================#
function SAMSUNGNvmeMain()
{
    local deviceType=$1
    local vendorType=$2
    local ctrlModel=$3
    local fwType=$4
    local fwVer=$5
    local fwToolsDir=$6
    local fwFiles=$7

    local fullPathFw="${FIRMWARE_PATH}/${fwFiles}"

    SAMSUNG_NVME_TOOL=${SCRIPT_PATH}"/"${fwToolsDir}"/"${NVME_TOOL_NAME}

    if [ ! -f "${SAMSUNG_NVME_TOOL}" -o ! -f "${fullPathFw}" ]
    then
        redFont "${SAMSUNG_NVME_TOOL} or ${fullPathFw} don't exist"
        return 0
    fi

    if [ $# -gt ${CFG_FILE_COLUMN_NUM} ]
    then
        local paraStartPos
        let paraStartPos=${CFG_FILE_COLUMN_NUM}+1
        for moreArgs in ${@:${paraStartPos}}
        do
            case "${moreArgs}" in
             "list")
               listSAMSUNGDiskDrive "${ctrlModel}" "${fwVer}" "${fwType}"
               ;;
             *)
               echo "no implement function :${moreArgs}"
               ;;
            esac
        done
        return 0
    fi

    case "${ctrlModel}" in
     "MZQL2960HCJR-00B7C"|"MZQL21T9HCJR-00B7C"|"MZQL23T8HCLS-00B7C"|"MZQL27T6HBLA-00B7C")
        if [ ${deviceType} = "harddisk" -a ${vendorType} = "SAMSUNG" ]
        then
            updateSAMSUNGDiskFW "${ctrlModel}" "${fullPathFw}" "${fwVer}" "${fwType}"
        else
            echo "Don't support ${deviceType} or ${vendorType}"
        fi
        ;;
      *)
         echo "Don't support disk model: ${ctrlModel}"
         ;;
    esac
}

SAMSUNGNvmeMain $@