#!/bin/bash

#升级fw工具
PMC_UPDATE_FW_TOOL_NAME="arcconf-linux"
#升级seeprom工具
PMC_UPDATE_SEEPROM_TOOL_NAME="ssflash_linux_64"


#定义SDPSx字典
declare -A SDPSx_dic
SDPSx_dic=(["0b27_00"]="SDPSA" ["0b27_01"]="SDPSB" ["0b45_00"]="SDPSA_L" ["0b45_01"]="SDPSB_L" ["0b29_00"]="SDPSA_I" ["0b29_01"]="SDPSB_I")
#定义RM24x字典
declare -A RM24x_dic
RM24x_dic=(["5445"]="RM241" ["5446"]="RM242" ["5447"]="RM243" ["544d"]="RM241B" ["544e"]="RM242B" ["544f"]="RM243B")

##===================================#
# 功能： PMC82XX信息查询
# 入参：配置文件 各列
#    $1  model    : 控制卡类型;
#    $2  filename ：发布的最新固件文件;
#    $3  newVer   : 建议或待升级的固件版本号;
#    $4  fwType   ：固件类型
# 输出：子卡总数，显示子卡详细信息： 设备类型，厂商，型号，固件类型，序列号，当前固件版本号，位置等信息
##===================================#
function listPmc82xxDriveDetails()
{
    local ctrlModel=$1
    local infoFile=$2
    local newVer=$3
    local fwType=$4
    if [ ${fwType} = "fw" ];then
        printf "%-10s %-10s %-20s %-10s %-16s %-8s %-3s\n"   Device  Verdor  Model FwType SerialNum  FwVer Slot >>pmc82xx.list
    fi
    OLD_IFS="$IFS"
    IFS=$'\n'
    while read myline
    do
        local pmc_card_type="unknown"
        pmc_card_type=$(echo $myline |awk '{print$1}')
        if [ ${pmc_card_type} = ${ctrlModel} ];then
            local CtrlId=$(echo $myline | awk '{print $2}')
            local FwVer=$($PMC_UPDATE_FW_TOOL getconfig $CtrlId ad | grep "Firmware" | awk -F ':' '{print $NF}' | awk -F '[' '{print $1}' | sed -e 's/^[ \t]*//g')
            local slot=$(echo $myline | awk '{print $3}')
            local SeepromVer=$(echo $myline | awk '{print $4}')
            local CtrlSN=$(echo $myline | awk '{print $5}')
            if [ ${fwType} = "fw" ];then
                local CtrlVer=$($PMC_UPDATE_FW_TOOL getconfig $CtrlId ad | grep "Firmware" | awk -F ':' '{print $NF}' | awk -F '[' '{print $1}' | sed -e 's/^[ \t]*//g')
                CtrlVer="V${CtrlVer}"
            elif [ ${fwType} = "seeprom" ];then
                local CtrlVer=$SeepromVer
            fi
            if [ "${CtrlVer}" = "${newVer}" ];then 
                printf "%-10s %-10s %-20s %-10s %-16s %-8s %-3s \n" "raid" "pmc" ${ctrlModel} ${fwType} ${CtrlSN} "${CtrlVer}" ${slot}  >>pmc82xx.list
            else
                if [ ${fwType} = "seeprom" ];then
                    printf "%-10s %-10s %-20s %-10s %-16s %-8s %-3s \n" "raid" "pmc" ${ctrlModel} ${fwType} ${CtrlSN} "${CtrlVer}" ${slot}  >>pmc82xx.list                    
                else
                    echo "need update" >>${NEED_UPDATE_FLAG}
                    printf "\e[31m%-10s %-10s %-20s %-10s %-16s %-8s %-3s \e[0m\n" "raid" "pmc" ${ctrlModel} ${fwType} ${CtrlSN}  "${CtrlVer}" ${slot}  >>pmc82xx.list
                fi               
            fi
        else
            continue
        fi
    done < $infoFile
    IFS="$OLD_IFS"
    # echo -e ""     >>pmc82xx.list
}

##===================================#
# 功能： PMC82XX固件信息查询公共部分
# 入参：
#     $1  result_file ：结果文件;
# pmc板载卡获取info：
#Request:
#ipmitool -I lan -H 192.168.5.7 -U zteroot -P Superuser9! raw 0x34 0x36 0x12 0x0 0x02 0x1 0x1 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff (槽位号1)     
#Response:
#12 00 02 01 46 54 00 00 ff ff ff ff ff ff ff ff ff ff ff ff ff ff ff 05
#（byte5-6为boardID：RM241为0x5445;RM242为0x5446;RM243为0x5447;byte7为PcbId;byte8为BomId）
# SDPSA:0x0b27 SDPSA_L:0x0b45 SDPSA_I:0x0b29
##===================================#
function get_pmc_info()
{
    local result_file=$1
    if [ ! -f ${result_file} ]
    then
        echo "scaning the pmc raid card in server......."
        $PMC_UPDATE_FW_TOOL list >pmcinfo.txt
        cat pmcinfo.txt |while read line
        do
            Flag2100=$(echo $line|grep 'IOC2100'|wc -l)
            Flag3100=$(echo $line|grep 'ROC3100'|wc -l)
            FlagRS242_18i_4G=$(echo $line |grep -c "RS242-18i 4G")
            FlagRS243_18i=$(echo $line |grep -c "RS243-18i")
            FlagRS241_18i_2G=$(echo $line |grep -c "RS241-18i 2G")
            Flag3152_8i=$(echo $line|grep '3152-8i'|wc -l)
            Flag3154_16i=$(echo $line|grep '3154-16i'|wc -l)
            PMC_PCIE_SLOT="onboard"
            # 自研卡
            if [ ${Flag2100} -ne 0 ] || [ ${Flag3100} -ne 0 ];then
                CtrlId=$(echo $line | awk -F ':' '{print $1}'| awk '{print $NF}')
                slot=$(echo $line | awk -F ',' '{print $2}' | awk '{print $NF}')
                SeepromVer="NA"
                CtrlSN=$(echo $line | awk -F ',' '{print $(NF-1)}' | sed -e 's/^[ \t]*//g')
                CardType="unknown"
                if [ ${FlagRS242_18i_4G} -eq 0 ] && [ ${FlagRS241_18i_2G} -eq 0 ] && [ ${FlagRS243_18i} -eq 0 ];then
                    for i in {1..15};
                    do
                        result=$(ipmitool raw 0x34 0x36 0x12 0 2 1 ${i} 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff 0xff)
                        board_id_byte6=$(echo ${result} | grep "12 00 02 01"| sed -e 's/^[ \t]*//g'|awk '{print$6}')
                        board_id_byte5=$(echo ${result} | grep "12 00 02 01"| sed -e 's/^[ \t]*//g'|awk '{print$5}')
                        bom_id=$(echo ${result} | grep "12 00 02 01"| sed -e 's/^[ \t]*//g'|awk '{print$8}')
                        if [ ! -z ${board_id_byte6} ] && [ ${board_id_byte6} = "0b" ];then
                            CardType=${SDPSx_dic["${board_id_byte6}${board_id_byte5}_${bom_id}"]}
                            if [ ! -z ${CardType} ];then
                                break
                            fi
                        fi
                        if [ ! -z ${board_id_byte6} ] && [ ${board_id_byte6} = "54" ];then
                            CardType=${RM24x_dic["${board_id_byte6}${board_id_byte5}"]}
                            if [ ! -z ${CardType} ];then
                                break
                            fi
                        fi
                    done
                else
                    if [ ${FlagRS242_18i_4G} -ne 0 ];then
                        CardType="RS242"
                    elif [ ${FlagRS241_18i_2G} -ne 0 ];then
                        CardType="RS241"
                    elif [ ${FlagRS243_18i} -ne 0 ];then
                        CardType="RS243"
                    fi
                    get_pmc_pcie_slot ${CtrlId}
                fi
                echo ${CardType} ${CtrlId} ${slot} ${SeepromVer} ${CtrlSN} ${PMC_PCIE_SLOT} >>$result_file
            elif [ ${Flag3152_8i} -ne 0 ] || [ ${Flag3154_16i} -ne 0 ];then
                CtrlId=$(echo $line | awk -F ':' '{print $1}'| awk '{print $NF}')
                slot=$(echo $line | awk -F ',' '{print $2}' | awk '{print $NF}')
                SeepromVer="NA"
                CtrlSN=$(echo $line | awk -F ',' '{print $(NF-1)}' | sed -e 's/^[ \t]*//g')
                if [ ${Flag3152_8i} -ne 0 ];then
                    CardType="3152-8i"
                elif [ ${Flag3154_16i} -ne 0 ];then
                    CardType="3154-16i"
                fi
                get_pmc_pcie_slot ${CtrlId}
                echo ${CardType} ${CtrlId} ${slot} ${SeepromVer} ${CtrlSN} ${PMC_PCIE_SLOT} >>$result_file
            fi
        done
        rm -rf pmcinfo.txt
    fi
}

##===================================#
# 功能： 获取PMC卡的数量
# 入参：$1 卡类型
#       $2 pmc卡信息文件
# 输出：pmc_num_file
##===================================#
function get_pmc_num()
{
    local ctrlModel=$1
    local pmc_info_file=$2
    local total_num=0
    while read myline
    do
        local cardtype=$(echo $myline|awk '{print$1}')
        if [ ${cardtype} = ${ctrlModel} ];then
            let total_num=total_num+1
        fi
    done <${pmc_info_file}
    echo "$total_num" > "pmc_num_file"
}

##===================================#
# 功能： 获取PMC卡的pcie槽位
# 入参：$1 控制器id
# 输出：赋值全局变量
##===================================#
function get_pmc_pcie_slot()
{
    local ctrlid=$1
    dmidecode -t 9 | grep -E "ID|Bus Address" > "pmc_pcie_slot_info.tmp"
    local ctrl_pcie_info=$(${PMC_UPDATE_FW_TOOL} getconfig ${ctrlid} ad |grep "PCI Address" |awk '{print$NF}')
    local ctrl_bus_address="0000:$(echo $ctrl_pcie_info |awk -F: '{print$2}'):00.$(echo $ctrl_pcie_info |awk -F: '{print$NF}')"
    local line_number=$(cat "pmc_pcie_slot_info.tmp" |wc -l)
    # 两个一组数据
    local group_num=$[ $line_number / 2 ]
    local start_line=1
    for ((i=0;i<${group_num};i++))
    do
        let end_line=start_line+1
        local pcie_id=$(sed -n ${start_line},${end_line}p "pmc_pcie_slot_info.tmp"|grep "ID" | awk '{print $NF}')
        local bus_address=$(sed -n ${start_line},${end_line}p "pmc_pcie_slot_info.tmp" | grep "Bus Address" |awk '{print $NF}')
        if [ ${ctrl_bus_address} = ${bus_address} ];then
            PMC_PCIE_SLOT=${pcie_id}
            break
        fi
        let start_line=start_line+2
    done
}

##===================================#
# 功能： PMC82XX固件信息查询
# 入参：配置文件 各列
#     $1  model ：控制器类型;
#     $2  newVer: 建议或待升级的固件版本号;
#     $3  fwType: 固件类型
# 输出：子卡总数，显示子卡详细信息： 设备类型，厂商，型号，固件类型，序列号，当前固件版本号，位置等信息。
##===================================#
function listPmc82xxDrive()
{
    local ctrlModel=$1
    local newVer=$2
    local fwType=$3

    get_pmc_info "Pmc82xxInfo.tmp"
    if [ ! -f "Pmc82xxInfo.tmp" ];then
        echo "no support pmc card " > "Pmc82xxInfo.tmp"
    fi
    get_pmc_num "${ctrlModel}" "Pmc82xxInfo.tmp"
    local TotalNum=`cat "pmc_num_file"`
    rm -rf "pmc_num_file"
    if [ $fwType = "fw" ];then
        echo -e "====================pmc raid card model(${ctrlModel}) info===================== " >>pmc82xx.list
        printf "%-10s: %-8s\n"   TotalNum  ${TotalNum}    >>pmc82xx.list
    fi

    if [ ${TotalNum} -ne 0 ]
    then
        listPmc82xxDriveDetails "${ctrlModel}" "Pmc82xxInfo.tmp" "${newVer}" "$fwType"
    fi
}
##===================================#
# 功能： PMC 控制卡的固件升级结果检查
# 入参:
#       $1 : 升级结果文件
#       $2 : 固件类型
#       $3 : 待升级pmc raid卡个数
##===================================#
function pmc82xxUpdateCheck()
{
    local pmc82xxresult=$1
    local fwType=$2
    local pmcNum=$3
    local ctrlModel=$4
    updateNum=$(cat $pmc82xxresult | wc -l)
    if [ $updateNum -eq 0 ];then
        return 0
    fi
    fwSucced=$(cat $pmc82xxresult | grep $ctrlModel | grep "fw update OK" | wc -l)
    seepromSucced=$(cat $pmc82xxresult | grep $ctrlModel | grep "seeprom update OK" | wc -l)
    fwFailed=$(cat $pmc82xxresult | grep $ctrlModel | grep "fw update failed" | wc -l)
    seepromFailed=$(cat $pmc82xxresult | grep $ctrlModel | grep "seeprom update failed" | wc -l)
    if [ $fwType = "fw" ];then
        if [ $fwFailed -eq 0 ];then
            greenFont "total pmc card($ctrlModel) num:$pmcNum,fw succeed num：$fwSucced,fw failed num：$fwFailed" | tee -a pmc82xx.result
        else
            redFont "total pmc card($ctrlModel) num:$pmcNum,fw succeed num：$fwSucced,fw failed num：$fwFailed,please Check!" | tee -a pmc82xx.result
            while read myline
            do
                fwinfo=$(echo $myline | grep fw | wc -l)
                if [ fwinfo != 0 ];then               
                   redFont "$myline" | tee -a pmc82xx.result
                fi
            done <$pmc82xxresult
        fi
    elif [ ${fwType} = "seeprom_rm24x" ];then
        if [ $seepromSucced -eq $pmcNum ];then
            greenFont "total pmc card($ctrlModel) num:$pmcNum,seeprom succeed num:$seepromSucced,seeprom failed num:$seepromFailed" | tee -a pmc82xx.result
        else
            redFont "total pmc card($ctrlModel) num:$pmcNum,seeprom succeed num:$seepromSucced,seeprom failed num:$seepromFailed,please Check!" | tee -a pmc82xx.result
            while read myline
            do
                fwinfo=$(echo $myline | grep seeprom | wc -l)
                if [ fwinfo != 0 ];then               
                   redFont "$myline" | tee -a pmc82xx.result
                fi
            done <$pmc82xxresult
        fi
    fi
}

##===================================#
# 功能：校验多个卡是否存在slot槽位重复的情况
# slot槽位重复会导致seeprom烧错
##===================================#

function check_slot_info()
{
    local pmc_info_file=$1
    declare -A slot_dict
    while read myline
    do
        local slot=$(echo ${myline} |awk '{print $3}')
        local exist_flag=${slot_dict["${slot}"]}
        if [ -z ${exist_flag} ];then
            slot_dict["${slot}"]="exist"
        else
            return 1
        fi
    done < $pmc_info_file
    return 0
}


##===================================#
# 功能： PMC 控制卡的固件升级
# 入参:
#       $1 : Raid model:控制卡芯片类型
#       $2 : fw_file：待升级固件
#       $3 : newVersion：待升级固件版本信息
#       $4 : fwType：待升级固件类型
##===================================#
function updatePmc82xxFW()
{
    local ctrlModel=$1
    local fwFile=$2
    local newVersion=$3
    local fwType=$4
    get_pmc_info "updatePmc82xxFW.tmp"
    $PMC_UPDATE_FW_TOOL list >> ${UPDATE_LOG}
    if [ ! -f "updatePmc82xxFW.tmp" ];then
        echo "no support pmc card " > "updatePmc82xxFW.tmp"
    fi
    get_pmc_num "${ctrlModel}" "updatePmc82xxFW.tmp"
    local PmcContrlNum=$(cat "pmc_num_file")
    rm -rf "pmc_num_file"

    if [ ${PmcContrlNum} -eq 0 ]
    then 
        echo -e "Total raid card(${ctrlModel}) num: 0,success num : 0" |tee -a pmc82xx.result
        greenFont "no raid card (${ctrlModel}) in this server,no need to update" |tee -a pmc82xx.result
        updateResult $ctrlModel "--" "fw" "success" "--" "--" "0" "no card"
        return 0
    fi
    check_slot_info "updatePmc82xxFW.tmp"
    if [ $? -ne 0 ];then
        redFont "pmc card slot repeat, exit update"
        updateResult $ctrlModel "--" "fw" "failed" "--" "--" "0" "card slot repeat"
        return 1
    fi
    while read myline
    do
        local pmc_card_type="unknown"
        pmc_card_type=$(echo $myline | awk '{print$1}')
        if [ ${pmc_card_type} = ${ctrlModel} ];then
            CtrlId=$(echo $myline | awk '{print $2}')
            slot=$(echo $myline | awk '{print $3}')
            SeepromVer=$(echo $myline | awk '{print $4}')
            single_raid_pcie_slot=$(echo $myline | awk '{print $NF}')
            starttime=`date +'%Y-%m-%d %H:%M:%S'`
            start_seconds=$(date --date="$starttime" +%s)
            if [ ${fwType} = "seeprom" ];then
                local CtrlVer=$SeepromVer
            elif [ ${fwType} = "fw" ];then
                local CtrlVer=$($PMC_UPDATE_FW_TOOL getconfig $CtrlId ad | grep "Firmware" | awk -F ':' '{print $NF}' | awk -F '[' '{print $1}' | sed -e 's/^[ \t]*//g')
                CtrlVer="V${CtrlVer}"
            else
                redFont "raid card(${ctrlModel}) $CtrlId fwtype is error " | tee -a pmc82xx.result
                endtime=`date +'%Y-%m-%d %H:%M:%S'`
                end_seconds=$(date --date="$endtime" +%s)
                updatetime=$((end_seconds-start_seconds))
                updateResult $ctrlModel $CtrlId $fwType "failed" "--" $newVersion $updatetime
                continue
            fi
            if [ ${CtrlVer} = ${newVersion} ];then
                greenFont "raid card(${ctrlModel}) pcie_slot:${single_raid_pcie_slot} controller_id:$CtrlId $fwType is new version(${newVersion}) in this server,no need to update " |tee -a pmc82xx.result
                endtime=`date +'%Y-%m-%d %H:%M:%S'`
                end_seconds=$(date --date="$endtime" +%s)
                updatetime=$((end_seconds-start_seconds))
                updateResult $ctrlModel $CtrlId $fwType "success" $CtrlVer $newVersion $updatetime
                continue
            else
                if [ ${fwType} = "fw" ];then
                    updatestatus=`$PMC_UPDATE_FW_TOOL romupdate $CtrlId $fwFile noprompt |grep "Succeeded"|wc -l`
                    endtime=`date +'%Y-%m-%d %H:%M:%S'`
                    end_seconds=$(date --date="$endtime" +%s)
                    updatetime=$((end_seconds-start_seconds))
                    if [ 1 -eq $updatestatus ];then
                        echo "raid card($ctrlModel) $CtrlId $fwType update OK" >>pmc82xxresult.tmp
                        echo "raid card($ctrlModel) pcie_slot:${single_raid_pcie_slot} controller_id:$CtrlId $fwType $fwFile ${newVersion} update OK" >> ${UPDATE_LOG}
                        updateResult $ctrlModel $CtrlId $fwType "success" $CtrlVer $newVersion $updatetime
                        #greenFont "raid card($ctrlModel) $CtrlId $fwType update OK" | tee -a pmc82xx.result
                    else
                        echo "raid card($ctrlModel) $CtrlId $fwType update failed,please Check!" >>pmc82xxresult.tmp
                        echo "raid card($ctrlModel) pcie_slot:${single_raid_pcie_slot} controller_id:$CtrlId $fwType $fwFile ${newVersion} update failed,please Check!" >> ${UPDATE_LOG}
                        updateResult $ctrlModel $CtrlId $fwType "failed" $CtrlVer $newVersion $updatetime
                        #redFont "raid card($ctrlModel) $CtrlId $fwType update failed,please Check!" | tee -a pmc82xx.result
                    fi
                elif [ ${fwType} = "seeprom" ];then
                    $PMC_UPDATE_SEEPROM_TOOL --ctrl slot=$slot --flash-nvram   $fwFile  --preserve
                    if [ $? -eq 0 ];then
                        endtime=`date +'%Y-%m-%d %H:%M:%S'`
                        end_seconds=$(date --date="$endtime" +%s)
                        updatetime=$((end_seconds-start_seconds))
                        echo "raid card($ctrlModel) $CtrlId $fwType update OK" >>pmc82xxresult.tmp
                        echo "raid card($ctrlModel) pcie_slot:${single_raid_pcie_slot} controller_id:$CtrlId $fwType $fwFile ${newVersion} update OK" >> ${UPDATE_LOG}
                        updateResult $ctrlModel $CtrlId $fwType "success" $CtrlVer $newVersion $updatetime
                        #greenFont "raid card($ctrlModel) $CtrlId $fwType update OK" | tee -a pmc82xx.result
                    else
                        endtime=`date +'%Y-%m-%d %H:%M:%S'`
                        end_seconds=$(date --date="$endtime" +%s)
                        updatetime=$((end_seconds-start_seconds))
                        echo "raid card($ctrlModel) $CtrlId $fwType update failed,please Check!" >>pmc82xxresult.tmp
                        echo "raid card($ctrlModel) pcie_slot:${single_raid_pcie_slot} controller_id:$CtrlId $fwType $fwFile ${newVersion} update failed,please Check!" >> ${UPDATE_LOG}
                        updateResult $ctrlModel $CtrlId $fwType "failed" $CtrlVer $newVersion $updatetime
                        #redFont "raid card($ctrlModel) $CtrlId $fwType update failed, please Check!" | tee -a pmc82xx.result
                    fi
                fi
            fi
        else
            continue
        fi
    done < updatePmc82xxFW.tmp
    if [ -f "pmc82xxresult.tmp" ];then
        pmc82xxUpdateCheck "pmc82xxresult.tmp" "$fwType" "$PmcContrlNum" "$ctrlModel"
        rm -rf "pmc82xxresult.tmp"
    fi
}

##===================================#
# 功能： PMC82XX的固件升级入口
# 入参：配置文件 各列
#     $1  device ; $2 vender ;$3 Model 
#     $4  FwType ; $5 FwVer  ;$6 固件升级的工具路径
#     $7  固件的版本文件
##===================================#
function pmc82xxMain()
{
    local contrlModel=$3
    local fwType=$4
    local fwVer=$5
    local fwToolsDir=$6
    local fwFiles=$7
    local deviceType=$1
    local vendorType=$2

    local fullPathFw="${FIRMWARE_PATH}/${fwFiles}"

    PMC_UPDATE_FW_TOOL=${SCRIPT_PATH}"/"${fwToolsDir}"/"${PMC_UPDATE_FW_TOOL_NAME}

    PMC_UPDATE_SEEPROM_TOOL=${SCRIPT_PATH}"/"${fwToolsDir}"/"${PMC_UPDATE_SEEPROM_TOOL_NAME}
    #PMC卡PCIE槽位
    PMC_PCIE_SLOT="onboard"

    if [ ! -f "${PMC_UPDATE_FW_TOOL}" -o ! -f "${fullPathFw}" ]
    then
        redFont " ${PMC_UPDATE_FW_TOOL} or ${fullPathFw} dont exist"
        return 0
    fi

    #非配置文件中的参数处理
    if [ $# -gt ${CFG_FILE_COLUMN_NUM} ]
    then
       local paraStartPos
       let paraStartPos=${CFG_FILE_COLUMN_NUM}+1
       for moreArgs in ${@:${paraStartPos}}
       do
          case "${moreArgs}" in
             "list")
              listPmc82xxDrive "${contrlModel}" "${fwVer}" "${fwType}"
              ;;
              *)
              echo "no implement function :${moreArgs}"
              ;;
          esac
       done
       return 0
    fi

    #固件升级
    case "$contrlModel" in
    "SDPSA"|"SDPSB"|"SDPSA_L"|"SDPSB_L"|"SDPSA_I"|"SDPSB_I"|"RM241"|"RM242"|"RM243"|"3152-8i"|"3154-16i"|"RS242"|"RS243"|"RS241"|"RM241B"|"RM242B"|"RM243B")
        if [ $deviceType = "raid" -a $vendorType = "PMC" ];then
            updatePmc82xxFW "${contrlModel}" "${fullPathFw}" "${fwVer}" "${fwType}"
        else
            echo "don't support $deviceType or $vendorType"
        fi
        ;;
      *)
        echo "don't support pmc raid card model :${contrlModel}"
        ;;
    esac
}

pmc82xxMain $@