# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

from abc import ABC, ABCMeta
import inspect


class FactoryMeta(ABCMeta):
    """Metaclass for defining factory pattern on class registration approach."""
    def __init__(cls, name, bases, namespace):
        super().__init__(name, bases, namespace)
        if issubclass(cls, Product):
            if cls is not Product and cls.subclasses is None:
                # create a new mapping for each subclass hierarchy of a Product
                cls.subclasses = {}
            # register only concrete products
            if cls.__product_type__ and not inspect.isabstract(cls):
                # check duplicate registration for a product type
                if cls.__product_type__ in cls.subclasses:
                    raise TypeError('duplicate registration for product ID '
                                    '{}'.format(cls.__product_type__))
                # register product ID to corresponded class object
                cls.subclasses[cls.__product_type__] = cls


class Product(ABC):
    """A base class that represents a factory product.

    :cvar str __product_type__: A unique product ID for class registration
    :cvar dict[str, Product] subclasses: A mapping between product ID to class
    """
    __product_type__ = None
    subclasses = None

    @classmethod
    def __subclasshook__(cls, subclass):
        if cls is Product:
            return hasattr(subclass, '__product_type__')
        return NotImplemented

    @classmethod
    def create(cls, product_type, *args, **kwargs):
        """Creates a new product object by a specified product type.

        :param product_type: The type of the product to be created
        :param args: Arbitrary positional arguments for specific product
        :param kwargs: Arbitrary keyword arguments for specific product
        :return: A new product instance.
        :rtype: Product
        """
        # Get the required class by a specified product type
        product_cls = cls.subclasses.get(product_type, None)
        if product_cls is None:
            raise TypeError('product of type \'{}\' is not supported'.format(
                product_type))
        elif issubclass(product_cls, cls) is False:
            raise TypeError('\'{}\' is not a product of \'{}\''.format(
                product_cls.__name__, cls.__name__))
        # create a new instance
        return product_cls(*args, **kwargs)

    @classmethod
    def get_subclasses_types(cls):
        """Returns a list of all registered product types."""
        return list(cls.subclasses or {})
