# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

from abc import abstractmethod
import errno
import os

from ..factory import Product, FactoryMeta

__all__ = ['PacketDumper']


class PacketDumper(Product, metaclass=FactoryMeta):
    """An abstract class that represents packet dumper.

    :param str path: The output file path for writing packets
    """

    def __init__(self, path, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.path = path

    @abstractmethod
    def dump(self, packets):
        """Dumps a list of packets into a file

        :param list[Packet] packets: A list of packets to be saved
        """
        try:
            # makes all intermediate-level directories for the leaf directory.
            leaf_dir = os.path.dirname(self.path)
            os.makedirs(leaf_dir, exist_ok=True)
            assert os.access(leaf_dir, os.W_OK)
        except AssertionError:
            raise PermissionError(errno.EACCES, 'Permission denied', self.path)
        except PermissionError as err:
            err.filename = self.path
            raise err

    @classmethod
    def get_packets_buffers(cls, packets):
        """Returns list of packets buffers from the given packets.

        :param packets: List of packets
        :return: packets buffers
        :rtype: List[bytes]
        """
        return [packet.data for packet in packets]
