# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

import os
import struct
import sys

from .provider import Provider
from ..constants import ProviderType, SegmentType
from ..errors import FetchICMAddressError


mft_py_dir = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
sys.path.append(os.path.join(mft_py_dir, 'resourcedump'))
from resourcedump.fetchers.ResourceDumpFetcher import ResourceDumpFetcher  # noqa
from regaccess import RegAccException  # noqa

__all__ = ['ResourceDump']


class ResourceDump(Provider):
    """A concrete class that represents resource dump provider.

    :param str device: The MST device to act operate on
    """
    __product_type__ = ProviderType.RESOURCEDUMP.value

    def __init__(self, device, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.device = device

    @classmethod
    def get_query_params(cls, icm_addr):
        """Returns query parameters from an ICM address object.

        :param ICMAddress icm_addr: An ICM address object
        :return: Query parameters for resource dump fetcher.
        :rtype: tuple
        """
        if SegmentType.STE_ROOT.value == icm_addr.seg_type:
            num_of_obj_1, vhca_id = None, None
        elif SegmentType.STE_GLOBAL_RNG.value == icm_addr.seg_type:
            num_of_obj_1, vhca_id = 1, None
        else:
            num_of_obj_1, vhca_id = 1, icm_addr.gvmi
        return hex(icm_addr.seg_type), vhca_id, icm_addr.index, num_of_obj_1

    def fetch(self, icm_addr):
        """Fetch steering raw data by an ICM address object.

        :param ICMAddress icm_addr: The ICM address to be look for
        :return: Raw STE data on little-endian byte order.
        :rtype: bytes
        """
        fetcher = ResourceDumpFetcher(self.device)
        segment, vhca_id, index1, num_of_obj_1 = self.get_query_params(icm_addr)
        try:
            segments = fetcher.fetch_data(segment=segment,
                                          depth=1,
                                          index1=index1,
                                          index2=None,
                                          numOfObj1=num_of_obj_1,
                                          numOfObj2=None,
                                          vHCAid=vhca_id,
                                          mem=None,
                                          fast_mode=None)
            # filter segments by ICM address's segment type
            segments = [s for s in segments if s.get_type() == segment]
            # fetch the steering raw data (64 bytes) from a segment
            data = segments[0].get_data()[4:]
        except (RegAccException, IndexError):
            raise FetchICMAddressError(icm_addr)
        else:
            # return the raw STE data on little-endian byte order
            return b''.join(struct.pack('<I', num) for num in data)
