# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

import ctypes
import os
import sys

from .stetypes import icm_obj_addr, packet_attr, steering_crawler, \
    steering_crawler_attr

try:
    mft_py_dir = os.path.dirname(os.path.dirname(__file__))
    stelib = ctypes.CDLL(os.path.join(mft_py_dir, 'stelib.so'), use_errno=True)
except OSError as err:
    sys.exit('Failed to load shared library stelib.so: {}'.format(err))


################################################################################
# Python functions
################################################################################
def create_icm_object_address():
    """Creates a new pointer to an empty ICM object address.

    :return: A new pointer to an ICM object address.
    :rtype: LP_icm_obj_addr
    """
    return ctypes.pointer(icm_obj_addr())


def create_packet_attribute(buffer, metadata):
    """Creates a new pointer to packet attribute.

    :param buffer: A buffer represents raw packet data
    :param Dict metadata: Attributes describing the input packet information
    :return: A new pointer to packet attribute.
    :rtype: LP_packet_attr
    """
    pkt_attr = packet_attr()
    # common attributes
    pkt_attr.packet_buf = ctypes.create_string_buffer(buffer)
    pkt_attr.packet_len = len(buffer)
    pkt_attr.flags = metadata.get('flags', 0)
    pkt_attr.vhca_port_num = metadata.get('phy_port', 0)
    # egress attributes
    pkt_attr.packet_src.egress.reg_a_value = metadata.get('reg_a_value', 0)
    pkt_attr.packet_src.egress.src_gvmi = metadata.get('virtual_hca_id', 0)
    pkt_attr.packet_src.egress.qpn = metadata.get('sqn', 0)
    pkt_attr.packet_src.egress.qp_flags = metadata.get('qp_flags', 0)
    return ctypes.pointer(pkt_attr)


def create_ste_crawler_attribute(verbosity=0):
    """Creates a new pointer to steering crawler attribute

    :param int verbosity: Specify the output verbosity level to use
    :return: A new pointer to steering crawler attribute.
    :rtype: LP_steering_crawler_attr
    """
    session_attr = steering_crawler_attr(verbosity_level=verbosity)
    return ctypes.pointer(session_attr)


################################################################################
# C functions
################################################################################
# create a packet steering crawler session
steering_crawler_create_analyze_packet = \
    stelib.steering_crawler_create_analyze_packet
steering_crawler_create_analyze_packet.restype = \
    ctypes.POINTER(steering_crawler)

# iterator analyze logic of ste for a given packet
steering_crawler_iterator = stelib.steering_crawler_iterator
steering_crawler_iterator.restype = ctypes.c_int

# destroy all resources associated with a packet steering_crawler session
steering_crawler_destroy = stelib.steering_crawler_destroy
steering_crawler_destroy.restype = None
