#!/bin/bash

#升级fw工具
NETSWIFT_UPDATE_FW_TOOL_NAME="wxtool_x86"

declare -A dev_model
dev_model=(["8088_0101_8088_c201"]="WX1860A2")

##===================================#
# 功能： Netswift网卡信息查询
# 入参:
#       $1: ctrlModel      网卡的型号
#       $2: infoFile       详细信息文件
#       $2: fwType         固件类型
#       $3: fwNewVer       固件版本
##===================================#
function listNetswiftCardDetails()
{
    local ctrlModel=$1
    local infoFile=$2
    local fwNewVer=$3
    local fwType=$4

    if [ ${fwType} = "default" ];then
        printf "%-10s %-10s %-20s %-10s %-16s %-8s %-3s\n"   Device Verdor BusInfo Model SerialNum FwType FwVer >> netswift.list
    fi

    OLD_IFS="${IFS}"
    IFS=$'\n'

    while read myline
    do
        local bus_info=$(echo $myline | awk '{print $1}')
        local cardType=$(echo $myline | awk '{print $2}')
        if [ "x${cardType}" = "x${ctrlModel}" ]
        then
            local cardCurVer=`${NETSWIFT_UPDATE_FW_TOOL} show -s ${bus_info} -i | grep "fw version" | awk -F ':' '{print $2}' | sed -e 's/\t//g' `
            if [ ${fwType} = "default" ]
            then
                if [ "x${cardCurVer}" = "x${fwNewVer}" ]
                then
                    printf "\e[37m%-10s %-10s %-20s %-10s %-16s %-8s %-3s \e[0m\n" "nic" "netswift" ${bus_info} ${ctrlModel} "--" ${fwType} ${fwNewVer} >> netswift.list
                else
                    echo "need update" >> ${NEED_UPDATE_FLAG}
                    printf "\e[31m%-10s %-10s %-20s %-10s %-16s %-8s %-3s \e[0m\n" "nic" "netswift" ${bus_info} ${ctrlModel} "--" ${fwType} ${fwNewVer} >> netswift.list
                fi
            fi
        fi
    done < ${infoFile}
    IFS="${OLD_IFS}"
}

##===================================#
# 功能： Netswift网卡固件信息查询
# 入参:
#       $1: ctrlModel   网卡的型号
#       $2: fwNewVer    固件类型
#       $3: fwType      固件版本
##===================================#
function listNetswiftCardDrive()
{
    local ctrlModel=$1
    local fwNewVer=$2
    local fwType=$3

    rm -f "netswiftCardInfo.tmp"
    echo "scanning the netswift nic card in server......"
    get_netswift_info "${ctrlModel}" "netswiftCardInfo.tmp"
    if [ ! -f "netswiftCardInfo.tmp" ]
    then
        echo "No this model netswift card found" > netswiftCardInfo.tmp
    fi

    local total_num=$(cat netswiftCardInfo.tmp | grep "${ctrlModel}" | wc -l)

    if [ ${fwType} = "default" ]
    then
        echo -e "====================pmc raid card model(${ctrlModel}) info=====================" >> netswift.list
        printf "%-10s: %-8s\n"   TotalNum  ${total_num}    >> netswift.list
    fi
    if [ ${total_num} -ne 0 ]
    then
        listNetswiftCardDetails "${ctrlModel}" "netswiftCardInfo.tmp" "${fwNewVer}" "${fwType}"
    fi
}

##===================================#
# 功能： Netswift网卡的固件升级结果检查
# 入参:
#       $1: netswiftResult.tmp   升级结果文件
#       $2: fwType           固件类型
#       $3: need_update_num  待升级netswift nic卡个数
#       $4: ctrlModel        卡的型号
#       $5: fwNewVer         固件版本
#       $6: updateTime       升级时间
##===================================#
function netswiftUpdateCheck()
{
    local netswiftResultFile=$1
    local fwType=$2
    local needUpdateNum=$3
    local ctrlModel=$4
    local fwNewVer=$5
    local updateTime=$6

    local fwSuccedNum=$(cat ${netswiftResultFile} | grep "default update succeed" | wc -l)
    local fwFailedNum=$(cat ${netswiftResultFile} | grep "default update failed" | wc -l)
    local fwUpdateNone=$(cat ${netswiftResultFile} | grep "default update 0" | wc -l)

    if [ ${fwType} = "default" ]
    then
        if [ ${fwSuccedNum} -eq ${needUpdateNum} ]
        then
            greenFont "Total nic card(${ctrlModel}) need update num:${needUpdateNum}, fw succeed num:${fwSuccedNum}, failed num:${fwFailedNum}" | tee -a netswift.result
            updateResult "${ctrlModel}" "all" "${fwType}" "success" "--" "${fwNewVer}" "${updateTime}"
        else
            if [ ${fwUpdateNone} -ne 0 ]
            then
                redFont "Please Check, there is 0 card are upgrade, use the tool ethtool and wxtool check firmware version" | tee -a netswift.result
            fi
            redFont "Total nic card(${ctrlModel}) need update num:${needUpdateNum}, fw succeed num:${fwSuccedNum}, failed num:${fwUpdateNone}, please Check" | tee -a netswift.result
            updateResult "${ctrlModel}" "all" "${fwType}" "failed" "--" "${fwNewVer}" "${updateTime}"
        fi
    fi
}

##===================================#
# 功能：NETSWIFT卡信息收集公共函数
# 入参：配置文件各列
#    $1: ctrlModel     网讯网卡类型
#    $2: result_file   netswift卡信息文件
##===================================#
function get_netswift_info()
{
    local ctrlModel=$1
    local result_file=$2

    #查看是否有可用的网讯网卡设备以及网卡信息
    if [ ! -f ${result_file} ]
    then
        echo "scanning the netswift nic card in server......"
        lspci | grep -E "Wangxun | 8088" > nicInfo.txt
        
        if [ -f nicInfo.txt ]
        then
            portNum=0
            nicInfoRow=$(cat nicInfo.txt | wc -l)

            for ((i=1; i < ${nicInfoRow}; i=i+${portNum}))
            do
                # 获取<BUS>:<DEVICE>
                bus_device=$(sed -n ${i}p nicInfo.txt | awk '{print $1}' | awk -F '.' '{print $1}')
                # 获取端口号<BUS>:<DEVICE>.<FUNCTION>
                bus_info=$(sed -n ${i}p nicInfo.txt | awk '{print $1}')
                # 获取device_id,vendor_id,svid,board_id用来判断网卡型号
                vendor_id=`lspci -s ${bus_info} -nm | awk '{print $3}' | sed -e 's/\"//g' `
                device_id=`lspci -s ${bus_info} -nm | awk '{print $4}' | sed -e 's/\"//g' `
                sub_ven_id=`lspci -s ${bus_info} -nm | awk '{print $6}' | sed -e 's/\"//g' `
                board_id=`lspci -s ${bus_info} -nm | awk '{print $7}' | sed -e 's/\"//g' `
                cardType=${dev_model["${vendor_id}_${device_id}_${sub_ven_id}_${board_id}"]}
                portNum=$(cat nicInfo.txt | grep -c "${bus_device}")
                if [ "x${cardType}" = "x${ctrlModel}" ]
                then
                    echo ${bus_info} ${cardType} >> ${result_file}
                fi
            done
            rm -f nicInfo.txt
        else
            redFont "Please check, no netswift NIC card in the server or no firmware in the NIC card"
            return 0
        fi
        
    fi
}

##===================================#
# 功能：NETSWIFT nic的固件升级
# 入参：
#       $1: ctrlModel  待升级nic卡型号
#       $2: fwFile     待升级固件文件
#       $3: fwNewVer   待升级固件版本
#       $4: fwType     待升级固件类型
##===================================#
function updateNetswiftFW()
{
    local ctrlModel=$1
    local fwFile=$2
    local fwNewVer=$3
    local fwType=$4

    rm -f "updateNetswiftFwInfo.tmp"
    chmod +x ${NETSWIFT_UPDATE_FW_TOOL}
    
    modprobe -v ngbe

    driverResultCheck=$(lsmod | grep "ngbe" | wc -l)
    if [ ${driverResultCheck} -ne 0 ]
    then
        greenFont "Load ngbe drive success" | tee -a netswift.result
    else
        redFont "Load ngbe drive failed, please Check!"
        updateResult "${ctrlModel}" "--" "${fwType}" "failed" "--" "--" "0" "load ngbe drive failed"
        return 0
    fi
    
    get_netswift_info "${ctrlModel}" "updateNetswiftFwInfo.tmp"

    check_fw_ver=`${NETSWIFT_UPDATE_FW_TOOL} check ${fwFile} | grep "fw version" | awk -F ':' '{print $2}' | sed -e 's/\t//g' `
    
    if [ ${check_fw_ver} != ${fwNewVer} ]
    then
        echo "The FwPackage show the fwVer is ${fwNewVer}, but actually the firmware version is ${check_fw_ver}" >> "fwVer.log"
    fi

    if [ ! -f "updateNetswiftFwInfo.tmp" ]
    then
        echo -e "Total nic card(${ctrlModel}) num: 0, success num: 0" | tee -a netswift.result
        greenFont "No nic card(${ctrlModel}) in this server, no need to update" | tee -a netswift.result
        updateResult "${ctrlModel}" "--" "${fwType}" "success" "--" "--" "0" "no card"
        return 0
    fi

    starttime=`date +'%Y-%m-%d %H:%M:%S'`
    start_seconds=$(date --date="$starttime" +%s)


    local total_num=$(cat updateNetswiftFwInfo.tmp | grep "${ctrlModel}" | wc -l)
    local already_update_num=0
    local need_update_num=0

    while read myline
    do
        local bus_info=$(echo $myline | awk '{print $1}')
        local cardType=$(echo $myline | awk '{print $2}')

        if [ "x${cardType}" = "x${ctrlModel}" ]
        then
            # 获取固件当前版本
            local cardCurVer=`${NETSWIFT_UPDATE_FW_TOOL} show -s ${bus_info} -i | grep "fw version" | awk -F ':' '{print $2}' | sed -e 's/\t//g' `
            if [ ${fwType} = "default" ]
            then
                if [ "x${cardCurVer}" = "x${fwNewVer}" ]
                then
                    let already_update_num=already_update_num+1
                else
                    # 千兆网卡的芯片分A和B，固件也分A和B,板载WX1860A2是B版本芯片
                    local check_img_ver=`${NETSWIFT_UPDATE_FW_TOOL} check ${fwFile} | grep "img_v" | awk -F ':' '{print $2}' | sed -e 's/\t//g' `
                    local chip_ver=`${NETSWIFT_UPDATE_FW_TOOL} show -s ${bus_info} -i | grep "chip_v" | awk -F ':' '{print $2}' | sed -e 's/\t//g' `
                    if [ "x${chip_ver}" != "x${check_img_ver}" ]
                    then
                        redFont "This nic card(${ctrlModel}) chip_ver is ${chip_ver}, img_ver is ${check_img_ver}, please check" | tee -a netswift.result
                        updateResult "${ctrlModel}" "--" "${fwType}" "failed" "--" "--" "0" "chip_verison type do not match"
                        return 0
                    fi

                    # 判断是千兆1口卡、千兆2口卡、千兆4口卡、万兆卡 -K[1、2、3、4]
                    case "${cardType}" in
                        "WX1860A2")
                            kindOfNIC=2
                        ;;
                    esac

                    # 升级
                    ${NETSWIFT_UPDATE_FW_TOOL} -F ${fwFile} -s ${bus_info} -K ${kindOfNIC} -I -S >& update_result.tmp
                    updateStatus=$(cat update_result.tmp | grep "upgrading is succeeded" | wc -l)
                    updateNone=$(cat update_result.tmp | grep "Only 0 cards are upgraded" | wc -l)

                    if [ ${updateStatus} -eq 1 ]
                    then
                        echo "Netswift nic card(${ctrlModel}) ${fwType} update succeed" > netswiftResult.tmp
                    else
                        if [ ${updateNone} -eq 1 ]
                        then
                            echo "Netswift nic card(${ctrlModel}) ${fwType} update 0 cards, please Check" > netswiftResult.tmp
                        else
                            echo "Netswift nic card(${ctrlModel}) ${fwType} update failed, please Check" > netswiftResult.tmp
                        fi
                        
                    fi
    
                fi
            fi
        fi
    done < updateNetswiftFwInfo.tmp

    let need_update_num=total_num-already_update_num
    

    endtime=`date +'%Y-%m-%d %H:%M:%S'`
    end_seconds=$(date --date="$endtime" +%s)
    updateTime=$((end_seconds-start_seconds))

    if [ ${need_update_num} -eq 0 ]
    then
        greenFont "All netswift nic card(${ctrlModel}) in this server is new Version(${fwNewVer}), no need to update" |tee -a netswift.result
        updateResult ${ctrlModel} "all" "${fwType}" "success" "${cardCurVer}" "${fwNewVer}" "${updateTime}"
        return 0
    fi

    if [ -f "netswiftResult.tmp" ]
    then
        netswiftUpdateCheck "netswiftResult.tmp" "${fwType}" "${need_update_num}" "${ctrlModel}" "${fwNewVer}" "${updateTime}"
        rm -f "netswiftResult.tmp"
    fi
}

##===================================#
# 功能：NETSWIFT nic的固件升级入口
# 入参：配置文件 各列
#     $1  deviceType 设备类型
#     $2  vendorType 厂商
#     $3  ctrlModel  固件型号
#     $4  fwType     待升级固件类型
#     $5  fwVer      固件版本号
#     $6  fwToolsDir 固件升级的工具路径
#     $7  fwFiles    固件的版本文件
##===================================#
function netswiftMain()
{
    local deviceType=$1
    local vendorType=$2
    local ctrlModel=$3
    local fwType=$4
    local fwVer=$5
    local fwToolsDir=$6
    local fwFiles=$7

    local fullPathFw="${FIRMWARE_PATH}/${fwFiles}"

    NETSWIFT_UPDATE_FW_TOOL=${SCRIPT_PATH}"/"${fwToolsDir}"/"${NETSWIFT_UPDATE_FW_TOOL_NAME}

    if [ ! -f "${NETSWIFT_UPDATE_FW_TOOL}" -o ! -f "${fullPathFw}" ]
    then
        redFont "${NETSWIFT_UPDATE_FW_TOOL} or ${fullPathFw} don't exist"
        updateResult "${ctrlModel}" "--" "${fwType}" "failed" "--" "--" "0" "update tool or firmware do not exist"
        return 0
    fi

    #非配置文件中参数处理
    if [ $# -gt ${CFG_FILE_COLUMN_NUM} ]
    then
        local paraStartPos
        let paraStartPos=${CFG_FILE_COLUMN_NUM}+1
        for moreArgs in ${@:${paraStartPos}}
        do
            case "${moreArgs}" in
             "list")
               listNetswiftCardDrive "${ctrlModel}" "${fwVer}" "${fwType}"
               ;;
             *)
               echo "no implement function: ${moreArgs}"
               ;;
            esac
        done
        return 0
    fi

    #固件升级
    case "${ctrlModel}" in
      "WX1860A2")
        if [ ${deviceType} = "nic" -a ${vendorType} = "NETSWIFT" ]
        then
            updateNetswiftFW "${ctrlModel}" "${fullPathFw}" "${fwVer}" "${fwType}"
        else
            echo "Don't support ${deviceType} or ${vendorType}"
        fi
        ;;
      *)
        echo "Don't support netswift nic card model:${ctrlModel}"
        ;;
    esac
}

netswiftMain $@